"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.cancelSubscription = exports.updateTeamMembers = exports.getBillingData = exports.getLatestUserData = exports.getLastUserData = exports.logOut = exports.hideLoginDialog = exports.showLoginDialog = exports.initializeAuthUi = exports.loginEvents = exports.RefreshRejectedError = exports.TokenRejectedError = void 0;
const _ = require("lodash");
const async_mutex_1 = require("async-mutex");
const events_1 = require("events");
const util_1 = require("@httptoolkit/util");
const jose_1 = require("jose");
const Auth0 = require("auth0-js");
const auth0_lock_1 = require("@httptoolkit/auth0-lock");
const auth0Dictionary = require('@httptoolkit/auth0-lock/lib/i18n/en').default;
const dedent = require("dedent");
const util_2 = require("./util");
const plans_1 = require("./plans");
const AUTH0_CLIENT_ID = 'KAJyF1Pq9nfBrv5l3LHjT9CrSQIleujj';
const AUTH0_DOMAIN = 'login.httptoolkit.tech';
// We read data from auth0 (via a netlify function), which includes
// the users subscription data, signed into a JWT that we can
// validate using this public key.
const AUTH0_DATA_PUBLIC_KEY = `
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAzRLZvRoiWBQS8Fdqqh/h
xVDI+ogFZ2LdIiMOQmkq2coYNvBXGX016Uw9KNlweUlCXUaQZkDuQBmwxcs80PEn
IliLvJnOcIA9bAJFEF36uIwSI/ZRj0faExanLO78cdIx+B+p69kFGlohQGzJmS1S
v/IYYu032hO+F5ypR+AoXn6qtGGLVN0zAvsvLEF3urY5jHiVbgk2FWD3FWMU3oBF
jEEjeSlAFnwJZgeEMFeYni7W/rQ8seU8y3YMIg2UyHpeVNnuWbJFFwGq8Aumg4SC
mCVpul3MYubdv034/ipGZSKJTwgubiHocrSBdeImNe3xdxOw/Mo04r0kcZBg2l/b
7QIDAQAB
-----END PUBLIC KEY-----
`.trim();
const auth0PublicKey = ((_a = globalThis === null || globalThis === void 0 ? void 0 : globalThis.crypto) === null || _a === void 0 ? void 0 : _a.subtle)
    ? (0, jose_1.importSPKI)(AUTH0_DATA_PUBLIC_KEY, 'RS256')
    : Promise.reject(new Error('WebCrypto not available in your browser. Auth is only possible in secure contexts (HTTPS).'));
class TokenRejectedError extends util_1.CustomError {
    constructor() {
        super('Auth token rejected');
    }
}
exports.TokenRejectedError = TokenRejectedError;
class RefreshRejectedError extends util_1.CustomError {
    constructor(response) {
        super(`Token refresh failed with: ${response.description}`);
    }
}
exports.RefreshRejectedError = RefreshRejectedError;
// Both always set as long as initializeAuthUi has been called.
let auth0Client;
let auth0Lock;
exports.loginEvents = new events_1.EventEmitter();
const initializeAuthUi = (options = {}) => {
    var _a, _b;
    auth0Client = new Auth0.Authentication({
        clientID: AUTH0_CLIENT_ID,
        domain: AUTH0_DOMAIN
    });
    auth0Lock = new auth0_lock_1.Auth0LockPasswordless(AUTH0_CLIENT_ID, AUTH0_DOMAIN, {
        configurationBaseUrl: 'https://cdn.eu.auth0.com',
        // Passwordless - email a code, confirm the code
        allowedConnections: ['email'],
        passwordlessMethod: 'code',
        auth: Object.assign({ 
            // Entirely within the app please
            redirect: false, 
            // Not used for redirects, but checked against auth0 config. Defaults to current URL, but
            // unfortunately that is a very large space, and each valid URL needs preconfiguring.
            redirectUrl: window.location.origin + '/', 
            // Required for passwordless (not normally, but it's reset when we use redirectUrl)
            responseType: options.refreshToken ? 'token' : 'token id_token' }, (options.refreshToken
            ? {
                // Include offline_access so that we get a refresh token
                params: { scope: 'openid email offline_access app_metadata' }
            } : {})),
        // UI config
        autofocus: true,
        allowAutocomplete: true,
        rememberLastLogin: (_a = options.rememberLastLogin) !== null && _a !== void 0 ? _a : true,
        closable: (_b = options.closeable) !== null && _b !== void 0 ? _b : true,
        theme: {
            primaryColor: '#e1421f',
            logo: 'https://httptoolkit.com/icon-600.png'
        },
        languageDictionary: Object.assign(auth0Dictionary, {
            title: 'Log in / Sign up',
            signUpTerms: dedent `
                No spam, this will only be used as your account login. By signing up, you accept
                the ToS & privacy policy.
            `
        })
    });
    // Forward auth0 events to the emitter
    [
        'authenticated',
        'unrecoverable_error',
        'authorization_error',
        'hide'
    ].forEach((event) => auth0Lock.on(event, (data) => exports.loginEvents.emit(event, data)));
    exports.loginEvents.on('user_data_loaded', () => auth0Lock.hide());
    // Synchronously load & parse the latest token value we have, if any
    try {
        // ! because actually parse(null) -> null, so it's ok
        tokens = JSON.parse(localStorage.getItem('tokens'));
    }
    catch (e) {
        console.log('Invalid token', localStorage.getItem('tokens'), e);
        exports.loginEvents.emit('app_error', 'Failed to parse saved auth token');
    }
};
exports.initializeAuthUi = initializeAuthUi;
const showLoginDialog = () => {
    if (!auth0Lock)
        throw new Error("showLoginDialog called before auth UI initialization");
    auth0Lock.show();
    // Login is always followed by either:
    // hide - user cancels login
    // user_data_loaded - everything successful
    // authorization_error - something (login/data loading/token request) goes wrong.
    return new Promise((resolve, reject) => {
        exports.loginEvents.once('user_data_loaded', () => resolve(true));
        exports.loginEvents.once('hide', () => resolve(false));
        exports.loginEvents.once('unrecoverable_error', reject);
        exports.loginEvents.on('authorization_error', (err) => {
            if (err.code === 'invalid_user_password')
                return; // Invalid login token, no worries
            else {
                console.log("Unexpected auth error", err);
                reject(err);
            }
        });
    });
};
exports.showLoginDialog = showLoginDialog;
const hideLoginDialog = () => auth0Lock === null || auth0Lock === void 0 ? void 0 : auth0Lock.hide();
exports.hideLoginDialog = hideLoginDialog;
const logOut = () => {
    exports.loginEvents.emit('logout');
};
exports.logOut = logOut;
let tokens; // Not initialized
const tokenMutex = new async_mutex_1.Mutex();
function setTokens(newTokens) {
    return tokenMutex.runExclusive(() => {
        tokens = newTokens;
        localStorage.setItem('tokens', JSON.stringify(newTokens));
    });
}
function updateTokensAfterAuth({ accessToken, refreshToken, expiresIn }) {
    setTokens({
        refreshToken,
        accessToken,
        accessTokenExpiry: Date.now() + (expiresIn * 1000)
    });
}
exports.loginEvents.on('authenticated', updateTokensAfterAuth);
exports.loginEvents.on('logout', () => setTokens(null));
// Must be run inside a tokenMutex
async function refreshToken() {
    if (!tokens)
        throw new Error("Can't refresh tokens if we're not logged in");
    if (tokens.refreshToken) {
        // If we have a permanent refresh token, we send it to Auth0 to get a
        // new fresh access token:
        return new Promise((resolve, reject) => {
            auth0Client.oauthToken({
                refreshToken: tokens.refreshToken,
                grantType: 'refresh_token'
            }, (error, result) => {
                if (error) {
                    if ([500, 403].includes(error.statusCode) &&
                        error.description && (error.description.includes('Grant not found') ||
                        error.description.includes('invalid refresh token'))) {
                        // Auth0 is explicitly rejecting our refresh token.
                        reject(new RefreshRejectedError(error));
                    }
                    else {
                        // Some other unknown error, might be transient/network issues
                        reject(error);
                    }
                }
                else {
                    tokens.accessToken = result.accessToken;
                    tokens.accessTokenExpiry = Date.now() + (result.expiresIn * 1000);
                    localStorage.setItem('tokens', JSON.stringify(tokens));
                    resolve(result.accessToken);
                }
            });
        });
    }
    else {
        // If not, we can still try to refresh the session, although with some
        // time limitations, so this might not always work.
        return new Promise((resolve, reject) => {
            auth0Lock.checkSession({}, (error, authResult) => {
                if (error)
                    reject(error);
                else {
                    resolve(authResult.accessToken);
                    updateTokensAfterAuth(authResult);
                }
            });
        });
    }
}
function getToken() {
    return tokenMutex.runExclusive(() => {
        if (!tokens)
            return;
        const timeUntilExpiry = tokens.accessTokenExpiry.valueOf() - Date.now();
        // If the token is expired or close (10 mins), refresh it
        let refreshPromise = timeUntilExpiry < 1000 * 60 * 10 ?
            refreshToken() : null;
        if (timeUntilExpiry > 1000 * 5) {
            // If the token is good for now, use it, even if we've
            // also triggered a refresh in the background
            return tokens.accessToken;
        }
        else {
            // If the token isn't usable, wait for the refresh
            return refreshPromise;
        }
    });
}
;
;
const anonUser = () => ({ featureFlags: [], banned: false });
const anonBillingAccount = () => ({ transactions: [], banned: false });
/*
 * Synchronously gets the last received user data, _without_
 * refreshing it in any way. After 7 days without a refresh
 * though, the result will change when the JWT expires.
 */
function getLastUserData() {
    try {
        const rawJwt = localStorage.getItem('last_jwt');
        const jwtData = getUnverifiedJwtPayload(rawJwt);
        if (jwtData) {
            // Validate what we can synchronously:
            if (!jwtData.exp)
                throw new Error('Missing expiry in JWT data');
            if ((jwtData.exp * 1000) < Date.now())
                throw new Error('Last JWT expired');
            // Async we do actually validate sigs etc, we just don't wait for it.
            getVerifiedJwtPayload(rawJwt, 'app').catch((e) => {
                localStorage.removeItem('last_jwt');
                console.log('Last JWT no longer valid - now cleared', e);
            });
        }
        return parseUserData(jwtData);
    }
    catch (e) {
        console.warn("Couldn't parse saved user data", e);
        return anonUser();
    }
}
exports.getLastUserData = getLastUserData;
/*
 * Get the latest valid user data we can. If possible, it loads the
 * latest data from the server. If that fails to load, or if it loads
 * but fails to parse, we return the latest user data.
 *
 * If there are no tokens available, or the latest data is expired,
 * this returns an empty (logged out) user.
 */
async function getLatestUserData() {
    try {
        const userRawJwt = await requestUserData('app');
        const jwtData = await getVerifiedJwtPayload(userRawJwt, 'app');
        const userData = await parseUserData(jwtData);
        localStorage.setItem('last_jwt', userRawJwt);
        return userData;
    }
    catch (e) {
        exports.loginEvents.emit('authorization_error', e);
        exports.loginEvents.emit('app_error', e);
        try {
            // Unlike getLastUserData, this does synchronously fully validate the data
            const lastUserData = localStorage.getItem('last_jwt');
            const jwtData = await getVerifiedJwtPayload(lastUserData, 'app');
            const userData = await parseUserData(jwtData);
            return userData;
        }
        catch (e) {
            console.log('Failed to validate last user JWT when updating', e);
            return anonUser();
        }
    }
}
exports.getLatestUserData = getLatestUserData;
async function getBillingData() {
    const userRawJwt = await requestUserData('billing');
    const jwtData = await getVerifiedJwtPayload(userRawJwt, 'billing');
    return parseBillingData(jwtData);
}
exports.getBillingData = getBillingData;
function getUnverifiedJwtPayload(jwt) {
    if (!jwt)
        return null;
    return (0, jose_1.decodeJwt)(jwt);
}
async function getVerifiedJwtPayload(jwt, type) {
    if (!jwt)
        return null;
    const decodedJwt = await (0, jose_1.jwtVerify)(jwt, await auth0PublicKey, {
        algorithms: ['RS256'],
        audience: `https://httptoolkit.tech/${type}_data`,
        issuer: 'https://httptoolkit.tech/'
    });
    return decodedJwt.payload;
}
function parseUserData(appData) {
    if (!appData)
        return anonUser();
    return {
        email: appData.email,
        subscription: parseSubscriptionData(appData),
        teamSubscription: appData.team_subscription
            ? parseSubscriptionData(appData.team_subscription)
            : undefined,
        featureFlags: appData.feature_flags || [],
        banned: !!appData.banned
    };
}
async function parseBillingData(billingData) {
    var _a, _b;
    if (!billingData)
        return anonBillingAccount();
    const transactions = (_b = (_a = billingData.transactions) === null || _a === void 0 ? void 0 : _a.map((transaction) => ({
        orderId: transaction.order_id,
        receiptUrl: transaction.receipt_url,
        sku: transaction.sku,
        createdAt: transaction.created_at,
        status: transaction.status,
        amount: transaction.amount,
        currency: transaction.currency
    }))) !== null && _b !== void 0 ? _b : null; // Null => transactions timed out upstream, not available.
    return {
        email: billingData.email,
        subscription: parseSubscriptionData(billingData),
        transactions,
        teamMembers: billingData.team_members,
        teamOwner: billingData.team_owner,
        lockedLicenseExpiries: billingData.locked_license_expiries,
        banned: !!billingData.banned
    };
}
function parseSubscriptionData(rawData) {
    var _a;
    const subscription = {
        status: rawData.subscription_status,
        plan: (_a = rawData.subscription_sku) !== null && _a !== void 0 ? _a : (0, plans_1.getSKUForPaddleId)(rawData.subscription_plan_id),
        quantity: rawData.subscription_quantity,
        expiry: rawData.subscription_expiry ? new Date(rawData.subscription_expiry) : undefined,
        updateBillingDetailsUrl: rawData.update_url,
        cancelSubscriptionUrl: rawData.cancel_url,
        lastReceiptUrl: rawData.last_receipt_url,
        canManageSubscription: !!rawData.can_manage_subscription
    };
    if (_.some(subscription) && !subscription.plan) {
        // No plan means no recognized plan, i.e. an unknown id. This should never happen,
        // but error reports suggest it's happened at least once.
        exports.loginEvents.emit('app_error', 'Invalid subscription data', rawData);
    }
    const optionalFields = [
        'lastReceiptUrl',
        'updateBillingDetailsUrl',
        'cancelSubscriptionUrl'
    ];
    const isCompleteSubscriptionData = _.every(_.omit(subscription, ...optionalFields), v => !_.isNil(v) // Not just truthy: canManageSubscription can be false on valid sub
    );
    // Use undefined rather than {} or partial data when there's any missing required sub fields
    return isCompleteSubscriptionData
        ? subscription
        : undefined;
}
async function requestUserData(type, options = {}) {
    const token = await getToken();
    if (!token)
        return '';
    const appDataResponse = await fetch(`${util_2.ACCOUNTS_API_BASE}/get-${type}-data`, {
        method: 'GET',
        headers: {
            'Authorization': `Bearer ${token}`
        }
    });
    if (!appDataResponse.ok) {
        console.log(`Received ${appDataResponse.status} loading ${type} data, with body: ${await appDataResponse.text()}`);
        if (appDataResponse.status === 401) {
            // We allow a single refresh+retry. If it's passed, we fail.
            if (options.isRetry)
                throw new TokenRejectedError();
            // If this is a first failure, let's assume it's a blip with our access token,
            // so a refresh is worth a shot (worst case, it'll at least confirm we're unauthed).
            return tokenMutex.runExclusive(() => refreshToken()).then(() => requestUserData(type, { isRetry: true }));
        }
        throw new Error(`Failed to load ${type} data`);
    }
    return appDataResponse.text();
}
async function updateTeamMembers(idsToRemove, emailsToAdd) {
    const token = await getToken();
    if (!token)
        throw new Error("Can't update team without an auth token");
    const appDataResponse = await fetch(`${util_2.ACCOUNTS_API_BASE}/update-team`, {
        method: 'POST',
        headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ idsToRemove, emailsToAdd })
    });
    if (!appDataResponse.ok) {
        const responseBody = await appDataResponse.text();
        console.log(`Received ${appDataResponse.status} updating team members: ${responseBody}`);
        throw new Error(responseBody || `Failed to update team members`);
    }
}
exports.updateTeamMembers = updateTeamMembers;
async function cancelSubscription() {
    const token = await getToken();
    if (!token)
        throw new Error("Can't cancel account without an auth token");
    const response = await fetch(`${util_2.ACCOUNTS_API_BASE}/cancel-subscription`, {
        method: 'POST',
        headers: {
            'Authorization': `Bearer ${token}`
        }
    });
    if (!response.ok) {
        throw new Error(`Unexpected ${response.status} response cancelling subscription`);
    }
}
exports.cancelSubscription = cancelSubscription;
//# sourceMappingURL=auth.js.map