"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.openNewCheckoutWindow = exports.goToCheckout = exports.prefetchCheckout = exports.prepareCheckout = void 0;
const util_1 = require("./util");
const getCheckoutUrl = (email, sku, source) => `${util_1.ACCOUNTS_API_BASE}/redirect-to-checkout?email=${encodeURIComponent(email)}&sku=${sku}&source=${window.location.hostname}&returnUrl=${encodeURIComponent(`https://httptoolkit.com/${source}-purchase-thank-you/`)}`;
// Create a link rel=prefetch (preloading a navigation) for a URL we're likely to open very shortly.
function prefetchPage(url) {
    const linkExists = !!document.head.querySelector(`link[href='${url}'][rel=prefetch]`);
    if (linkExists)
        return;
    const prerenderLink = document.createElement("link");
    prerenderLink.setAttribute("rel", "prefetch");
    prerenderLink.setAttribute("href", url);
    document.head.appendChild(prerenderLink);
}
/**
 * Forcing an initial fetch for this URL preps the server cache and speeds up the checkout
 * process a little (as we can do this while we load initial user data during checkout).
 *
 * This makes sense only when the real navigation will be in another context (e.g. the
 * system browser, when running in Electron). Otherwise prefetchCheckout does the same
 * thing server-side but helpfully also caches in the browser too.
 */
function prepareCheckout(email, sku, source) {
    fetch(getCheckoutUrl(email, sku, source), {
        redirect: 'manual' // We just prime the API cache, we don't navigate
    }).catch(() => { }); // Just an optimization - ignore errors
    return; // We don't return the promise - don't wait for this, just trigger it
}
exports.prepareCheckout = prepareCheckout;
;
/**
 * Prefetch the checkout, using a prefetch link in the HEAD. This should be used before
 * loading the checkout in the same context. If loading it elsewhere (the system browser)
 * then prepareCheckout should be used instead.
 */
function prefetchCheckout(email, sku, source) {
    prefetchPage(getCheckoutUrl(email, sku, source));
}
exports.prefetchCheckout = prefetchCheckout;
async function goToCheckout(email, sku, source) {
    window.location.assign(getCheckoutUrl(email, sku, source));
}
exports.goToCheckout = goToCheckout;
function openNewCheckoutWindow(email, sku, source) {
    window.open(getCheckoutUrl(email, sku, source), '_blank');
}
exports.openNewCheckoutWindow = openNewCheckoutWindow;
;
//# sourceMappingURL=checkout.js.map