"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractMockttp = void 0;
const common_tags_1 = require("common-tags");
const request_rule_builder_1 = require("./rules/requests/request-rule-builder");
const websocket_rule_builder_1 = require("./rules/websockets/websocket-rule-builder");
const types_1 = require("./types");
/**
 * @hidden
 */
class AbstractMockttp {
    constructor(options) {
        this.addRequestRule = (rule) => this.addRequestRules(rule).then((rules) => rules[0]);
        this.addWebSocketRule = (rule) => this.addWebSocketRules(rule).then((rules) => rules[0]);
        this.debug = options.debug || false;
        this.corsOptions = options.cors || false;
        this.recordTraffic = options.recordTraffic !== undefined
            ? options.recordTraffic
            : true;
        this.suggestChanges = options.suggestChanges !== undefined
            ? options.suggestChanges
            : true;
    }
    get proxyEnv() {
        return {
            HTTP_PROXY: this.url,
            HTTPS_PROXY: this.url
        };
    }
    urlFor(path) {
        return this.url + path;
    }
    forAnyRequest() {
        return new request_rule_builder_1.RequestRuleBuilder(this.addRequestRule);
    }
    forUnmatchedRequest() {
        return new request_rule_builder_1.RequestRuleBuilder(this.addRequestRule)
            .asPriority(types_1.RulePriority.FALLBACK);
    }
    forGet(url) {
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.GET, url, this.addRequestRule);
    }
    forPost(url) {
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.POST, url, this.addRequestRule);
    }
    forPut(url) {
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.PUT, url, this.addRequestRule);
    }
    forDelete(url) {
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.DELETE, url, this.addRequestRule);
    }
    forPatch(url) {
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.PATCH, url, this.addRequestRule);
    }
    forHead(url) {
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.HEAD, url, this.addRequestRule);
    }
    forOptions(url) {
        if (this.corsOptions) {
            throw new Error((0, common_tags_1.stripIndent) `
                Cannot mock OPTIONS requests with CORS enabled.

                You can disable CORS by passing { cors: false } to getLocal/getRemote, but this may cause issues ${''}connecting to your mock server from browsers, unless you mock all required OPTIONS preflight ${''}responses by hand.
            `);
        }
        return new request_rule_builder_1.RequestRuleBuilder(types_1.Method.OPTIONS, url, this.addRequestRule);
    }
    forJsonRpcRequest(match = {}) {
        return new request_rule_builder_1.RequestRuleBuilder(this.addRequestRule)
            .withJsonBodyIncluding({
            jsonrpc: '2.0',
            ...(match.method ? { method: match.method } : {}),
            ...(match.params ? { params: match.params } : {})
        });
    }
    forAnyWebSocket() {
        return new websocket_rule_builder_1.WebSocketRuleBuilder(this.addWebSocketRule);
    }
}
exports.AbstractMockttp = AbstractMockttp;
//# sourceMappingURL=mockttp.js.map