/// <reference types="node" />
import { EventEmitter } from 'events';
import { Duplex } from 'stream';
import { TypedError } from 'typed-error';
import { AdminPlugin, PluginClientResponsesMap, PluginStartParamsMap } from '../admin/admin-plugin-types';
import { SchemaIntrospector } from './schema-introspection';
import { AdminQuery } from './admin-query';
export declare class ConnectionError extends TypedError {
}
declare global {
    interface Response {
    }
}
export declare class RequestError extends TypedError {
    response: Response;
    constructor(message: string, response: Response);
}
export declare class GraphQLError extends RequestError {
    errors: Array<{
        message: string;
    }>;
    constructor(response: Response, errors: Array<{
        message: string;
    }>);
}
export declare type AdminClientEvent = 'starting' | 'started' | 'start-failed' | 'stopping' | 'stopped' | 'stream-error' | 'stream-reconnecting' | 'stream-reconnected' | 'stream-reconnect-failed' | 'subscription-error' | 'subscription-reconnecting';
export interface AdminClientOptions {
    /**
     * Should the client print extra debug information?
     */
    debug?: boolean;
    /**
     * The full URL to use to connect to a Mockttp admin server when using a
     * remote (or local but browser) client.
     *
     * When using a local server, this option is ignored.
     */
    adminServerUrl?: string;
    /**
     * Options to include on all client requests.
     */
    requestOptions?: {
        headers?: {
            [key: string]: string;
        };
    };
}
/**
 * Reset a remote admin server, shutting down all Mockttp servers controlled by that
 * admin server. This is equivalent to calling `client.stop()` for all remote
 * clients of the target server.
 *
 * This can be useful in some rare cases, where a client might fail to reliably tear down
 * its own server, e.g. in Cypress testing. In this case, it's useful to reset the
 * admin server completely remotely without needing access to any previous client
 * instances, to ensure all servers from previous test runs have been shut down.
 *
 * After this is called, behaviour of any previously connected clients is undefined, and
 * it's likely that they may throw errors or experience other undefined behaviour. Ensure
 * that `client.stop()` has been called on all active clients before calling this method.
 */
export declare function resetAdminServer(options?: AdminClientOptions): Promise<void>;
/**
 * A bare admin server client. This is not intended for general use, but can be useful when
 * building admin server plugins to mock non-HTTP protocols and other advanced use cases.
 *
 * For normal usage of Mockttp, you should use `Mockttp.getRemote()` instead, to get a Mockttp
 * remote client, which wraps this class with the full Mockttp API for mocking HTTP.
 *
 * This is part of Mockttp's experimental 'pluggable admin' API. It may change
 * unpredictably, even in minor releases.
 */
export declare class AdminClient<Plugins extends {
    [key: string]: AdminPlugin<any, any>;
}> extends EventEmitter {
    private adminClientOptions;
    private adminSessionBaseUrl;
    private adminServerStream;
    private subscriptionClient;
    private adminServerSchema;
    private adminServerMetadata;
    private debug;
    private running;
    constructor(options?: AdminClientOptions);
    private attachStreamWebsocket;
    private openStreamToMockServer;
    private prepareSubscriptionClientToAdminServer;
    private requestFromMockServer;
    private queryMockServer;
    start(pluginStartParams: PluginStartParamsMap<Plugins>): Promise<PluginClientResponsesMap<Plugins>>;
    isRunning(): boolean;
    get metadata(): NonNullable<PluginClientResponsesMap<Plugins>>;
    get schema(): SchemaIntrospector;
    get adminStream(): Duplex;
    stop(): Promise<void>;
    private requestServerStop;
    enableDebug: () => Promise<void>;
    reset: () => Promise<void>;
    sendQuery<Response, Result = Response>(query: AdminQuery<Response, Result>): Promise<Result>;
    sendQueries<Queries extends Array<AdminQuery<any>>>(...queries: [...Queries]): Promise<{
        [n in keyof Queries]: Queries[n] extends AdminQuery<any, infer R> ? R : never;
    }>;
    subscribe<Response, Result = Response>(query: AdminQuery<Response, Result>, callback: (data: Result) => void): Promise<void>;
    /**
     * List the names of the rule parameters defined by the admin server. This can be
     * used in some advanced use cases to confirm that the parameters a client wishes to
     * reference are available.
     *
     * Only defined for remote clients.
     */
    getRuleParameterKeys(): Promise<string[]>;
}
//# sourceMappingURL=admin-client.d.ts.map