"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeDestroyable = void 0;
/**
 * Makes a server 'destroyable': tracks all active server connections, and adds a
 * `.destroy()` method to the server, which destroys all active server connections
 * and then shuts the server down cleanly.
 *
 * The server is mutated (adding the new method) and also returned from this method
 * for convenience.
 *
 * `.destroy()` returns a promise, which you can wait on to ensure the server has
 * been fully destroyed.
 */
function makeDestroyable(server) {
    const connections = {};
    server.on('connection', function (conn) {
        const key = conn.remoteAddress + ':' + conn.remotePort;
        connections[key] = conn;
        conn.on('close', function () {
            delete connections[key];
        });
    });
    server.on('secureConnection', function (conn) {
        const key = conn.remoteAddress + ':' + conn.remotePort;
        connections[key] = conn;
        conn.on('close', function () {
            delete connections[key];
        });
    });
    return Object.assign(server, {
        destroy: () => {
            return new Promise((resolve, reject) => {
                server.close((err) => {
                    if (err)
                        reject(err);
                    else
                        resolve();
                });
                for (let key in connections) {
                    connections[key].destroy();
                }
            });
        }
    });
}
exports.makeDestroyable = makeDestroyable;
//# sourceMappingURL=index.js.map