"use strict";

var _immutable = _interopRequireDefault(require("immutable"));
var _data_utils = require("../../utils/data_utils");
var _index = require("../../core/index");
var _i18n = require("../../i18n");
var _testUtils = require("../testUtils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
var setResolvedConnection = function setResolvedConnection() {
  var _require;
  return (_require = require('core/index')).setResolvedConnection.apply(_require, arguments);
};
var setup = function setup() {
  var _require2;
  return (_require2 = require('core/index')).setup.apply(_require2, arguments);
};
var mockLock = 'm';
var mockSet;
var mockInit;
jest.mock('i18n', function () {
  return {
    initI18n: jest.fn(),
    html: function html() {
      for (var _len = arguments.length, keys = new Array(_len), _key = 0; _key < _len; _key++) {
        keys[_key] = arguments[_key];
      }
      return keys.join();
    }
  };
});
jest.mock('utils/data_utils', function () {
  return {
    dataFns: function dataFns() {
      return {
        get: jest.fn(),
        set: mockSet,
        init: mockInit
      };
    }
  };
});
describe('setup', function () {
  beforeEach(function () {
    mockInit = jest.fn();
    jest.resetModules();
  });
  it('default redirectUrl should not include location.hash', function () {
    (0, _testUtils.setURL)('https://test.com/path/#not-this-part');
    var options = {};
    setup('id', 'clientID', 'domain', options, 'hookRunner', 'emitEventFn');
    var _mockInit = mockInit,
      mock = _mockInit.mock;
    expect(mock.calls.length).toBe(1);
    var model = mock.calls[0][1].toJS();
    expect(model.auth.redirectUrl).toBe('https://test.com/path/');
  });
  it.only('default redirectUrl should work when `window.location.origin` is not available', function () {
    jsdom.reconfigure({
      url: 'https://test.com/path/#not-this-part'
    });
    var options = {};
    setup('id', 'clientID', 'domain', options, 'hookRunner', 'emitEventFn');
    var _mockInit2 = mockInit,
      mock = _mockInit2.mock;
    expect(mock.calls.length).toBe(1);
    var model = mock.calls[0][1].toJS();
    expect(model.auth.redirectUrl).toBe('https://test.com/path/');
  });
  it('should work with redirect:false and responseType:id_token', function () {
    var options = {
      auth: {
        redirect: false,
        responseType: 'id_token'
      }
    };
    setup('id', 'clientID', 'domain', options, 'hookRunner', 'emitEventFn', 'handleEventFn');
    var _mockInit3 = mockInit,
      mock = _mockInit3.mock;
    expect(mock.calls.length).toBe(1);
    var model = mock.calls[0][1].toJS();
    expect(model).toMatchSnapshot();
  });
  describe('clientBaseUrl', function () {
    it('should default to the specified domain', function () {
      var _mockInit4 = mockInit,
        mock = _mockInit4.mock;
      setup('id', 'clientID', 'my-tenant.us.auth0.com', {}, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.clientBaseUrl).toBe('https://my-tenant.us.auth0.com');
    });
    it('should use the clientBaseUrl option if given', function () {
      var _mockInit5 = mockInit,
        mock = _mockInit5.mock;
      setup('id', 'clientID', 'my-tenant.us.auth0.com', {
        clientBaseUrl: 'https://client-base-url.example.com',
        configurationBaseUrl: 'https://config-base-url.example.com',
        assetsUrl: 'https://assets-url.example.com'
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.clientBaseUrl).toBe('https://client-base-url.example.com');
    });
    it('should use configurationBaseUrl if given', function () {
      var _mockInit6 = mockInit,
        mock = _mockInit6.mock;
      setup('id', 'clientID', 'my-tenant.us.auth0.com', {
        configurationBaseUrl: 'https://config-base-url.example.com',
        assetsUrl: 'https://assets-url.example.com'
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.clientBaseUrl).toBe('https://config-base-url.example.com');
    });
    it('should use assetsUrl if given', function () {
      var _mockInit7 = mockInit,
        mock = _mockInit7.mock;
      setup('id', 'clientID', 'my-tenant.us.auth0.com', {
        assetsUrl: 'https://assets-url.example.com'
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.clientBaseUrl).toBe('https://assets-url.example.com');
    });
  });
  describe('tenantBaseUrl', function () {
    it('should default to domain URL when using auth0.com', function () {
      var _mockInit8 = mockInit,
        mock = _mockInit8.mock;
      setup('id', 'clientID', 'my-tenant.us.auth0.com', {
        __useTenantInfo: true
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.tenantBaseUrl).toBe('https://my-tenant.us.auth0.com/tenants/v1/my-tenant.js');
    });
    it('should default to domain URL when using a custom domain', function () {
      var _mockInit9 = mockInit,
        mock = _mockInit9.mock;
      setup('id', 'clientID', 'auth.my-tenant.com', {
        __useTenantInfo: true
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.tenantBaseUrl).toBe('https://auth.my-tenant.com/info-v1.js');
    });
    it('should use configurationBaseUrl if specified', function () {
      var _mockInit10 = mockInit,
        mock = _mockInit10.mock;
      setup('id', 'clientID', 'auth.my-tenant.com', {
        __useTenantInfo: true,
        configurationBaseUrl: 'https://config-base-url.com'
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.tenantBaseUrl).toBe('https://config-base-url.com/info-v1.js');
    });
    it('should use configurationBaseUrl with a custom tenant if specified', function () {
      var _mockInit11 = mockInit,
        mock = _mockInit11.mock;
      setup('id', 'clientID', 'auth.my-tenant.com', {
        __useTenantInfo: true,
        configurationBaseUrl: 'https://config-base-url.com',
        overrides: {
          __tenant: 'custom-tenant'
        }
      }, 'hookRunner', 'emitEventFn', 'handleEventFn');
      expect(mock.calls.length).toBe(1);
      var model = mock.calls[0][1].toJS();
      expect(model.tenantBaseUrl).toBe('https://config-base-url.com/tenants/v1/custom-tenant.js');
    });
  });
});
describe('setResolvedConnection', function () {
  beforeEach(function () {
    mockSet = jest.fn();
    jest.resetModules();
  });
  it('sets undefined when is called with undefined', function () {
    setResolvedConnection(mockLock, undefined);
    expect(mockSet.mock.calls.length).toBe(1);
    expect(mockSet.mock.calls[0]).toMatchSnapshot();
  });
  it('validates format', function () {
    expect(function () {
      return setResolvedConnection(mockLock, {});
    }).toThrowErrorMatchingSnapshot();
    expect(function () {
      return setResolvedConnection(mockLock, {
        type: 'foo'
      });
    }).toThrowErrorMatchingSnapshot();
    expect(function () {
      return setResolvedConnection(mockLock, {
        name: 'bar'
      });
    }).toThrowErrorMatchingSnapshot();
  });
  it('accepts only database connections', function () {
    expect(function () {
      return setResolvedConnection(mockLock, {
        type: 'foo',
        name: 'bar'
      });
    }).toThrowErrorMatchingSnapshot();
  });
  it('sets the connection', function () {
    setResolvedConnection(mockLock, {
      type: 'database',
      name: 'bar'
    });
    expect(mockSet.mock.calls.length).toBe(1);
    expect(mockSet.mock.calls[0]).toMatchSnapshot();
  });
  it('sets the connection as a Map instance', function () {
    setResolvedConnection(mockLock, {
      type: 'database',
      name: 'bar'
    });
    expect(mockSet.mock.calls.length).toBe(1);
    expect(_immutable.default.Map.isMap(mockSet.mock.calls[0][2])).toBe(true);
  });
});
describe('loginErrorMessage', function () {
  it('maps `password_expired` to `password_change_required`', function () {
    var result = (0, _index.loginErrorMessage)(mockLock, {
      code: 'password_expired'
    }, 'type');
    expect(result).toBe([mockLock, 'error', 'login', 'password_change_required'].join());
  });
});
