/*
 * Copyright 2021 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

import { isIPv4, isIPv6 } from 'net';
import { OrderedMap, type OrderedMapIterator } from '@js-sdsl/ordered-map';
import { ConnectivityState } from './connectivity-state';
import { Status } from './constants';
import { Timestamp } from './generated/google/protobuf/Timestamp';
import { Channel as ChannelMessage } from './generated/grpc/channelz/v1/Channel';
import { ChannelConnectivityState__Output } from './generated/grpc/channelz/v1/ChannelConnectivityState';
import { ChannelRef as ChannelRefMessage } from './generated/grpc/channelz/v1/ChannelRef';
import { ChannelTrace } from './generated/grpc/channelz/v1/ChannelTrace';
import { GetChannelRequest__Output } from './generated/grpc/channelz/v1/GetChannelRequest';
import { GetChannelResponse } from './generated/grpc/channelz/v1/GetChannelResponse';
import { sendUnaryData, ServerUnaryCall } from './server-call';
import { ServerRef as ServerRefMessage } from './generated/grpc/channelz/v1/ServerRef';
import { SocketRef as SocketRefMessage } from './generated/grpc/channelz/v1/SocketRef';
import {
  isTcpSubchannelAddress,
  SubchannelAddress,
} from './subchannel-address';
import { SubchannelRef as SubchannelRefMessage } from './generated/grpc/channelz/v1/SubchannelRef';
import { GetServerRequest__Output } from './generated/grpc/channelz/v1/GetServerRequest';
import { GetServerResponse } from './generated/grpc/channelz/v1/GetServerResponse';
import { Server as ServerMessage } from './generated/grpc/channelz/v1/Server';
import { GetServersRequest__Output } from './generated/grpc/channelz/v1/GetServersRequest';
import { GetServersResponse } from './generated/grpc/channelz/v1/GetServersResponse';
import { GetTopChannelsRequest__Output } from './generated/grpc/channelz/v1/GetTopChannelsRequest';
import { GetTopChannelsResponse } from './generated/grpc/channelz/v1/GetTopChannelsResponse';
import { GetSubchannelRequest__Output } from './generated/grpc/channelz/v1/GetSubchannelRequest';
import { GetSubchannelResponse } from './generated/grpc/channelz/v1/GetSubchannelResponse';
import { Subchannel as SubchannelMessage } from './generated/grpc/channelz/v1/Subchannel';
import { GetSocketRequest__Output } from './generated/grpc/channelz/v1/GetSocketRequest';
import { GetSocketResponse } from './generated/grpc/channelz/v1/GetSocketResponse';
import { Socket as SocketMessage } from './generated/grpc/channelz/v1/Socket';
import { Address } from './generated/grpc/channelz/v1/Address';
import { Security } from './generated/grpc/channelz/v1/Security';
import { GetServerSocketsRequest__Output } from './generated/grpc/channelz/v1/GetServerSocketsRequest';
import { GetServerSocketsResponse } from './generated/grpc/channelz/v1/GetServerSocketsResponse';
import {
  ChannelzDefinition,
  ChannelzHandlers,
} from './generated/grpc/channelz/v1/Channelz';
import { ProtoGrpcType as ChannelzProtoGrpcType } from './generated/channelz';
import type { loadSync } from '@grpc/proto-loader';
import { registerAdminService } from './admin';
import { loadPackageDefinition } from './make-client';

export type TraceSeverity =
  | 'CT_UNKNOWN'
  | 'CT_INFO'
  | 'CT_WARNING'
  | 'CT_ERROR';

interface Ref {
  kind: EntityTypes;
  id: number;
  name: string;
}

export interface ChannelRef extends Ref {
  kind: EntityTypes.channel;
}

export interface SubchannelRef extends Ref {
  kind: EntityTypes.subchannel;
}

export interface ServerRef extends Ref {
  kind: EntityTypes.server;
}

export interface SocketRef extends Ref {
  kind: EntityTypes.socket;
}

function channelRefToMessage(ref: ChannelRef): ChannelRefMessage {
  return {
    channel_id: ref.id,
    name: ref.name,
  };
}

function subchannelRefToMessage(ref: SubchannelRef): SubchannelRefMessage {
  return {
    subchannel_id: ref.id,
    name: ref.name,
  };
}

function serverRefToMessage(ref: ServerRef): ServerRefMessage {
  return {
    server_id: ref.id,
  };
}

function socketRefToMessage(ref: SocketRef): SocketRefMessage {
  return {
    socket_id: ref.id,
    name: ref.name,
  };
}

interface TraceEvent {
  description: string;
  severity: TraceSeverity;
  timestamp: Date;
  childChannel?: ChannelRef;
  childSubchannel?: SubchannelRef;
}

/**
 * The loose upper bound on the number of events that should be retained in a
 * trace. This may be exceeded by up to a factor of 2. Arbitrarily chosen as a
 * number that should be large enough to contain the recent relevant
 * information, but small enough to not use excessive memory.
 */
const TARGET_RETAINED_TRACES = 32;

/**
 * Default number of sockets/servers/channels/subchannels to return
 */
const DEFAULT_MAX_RESULTS = 100;

export class ChannelzTraceStub {
  readonly events: TraceEvent[] = [];
  readonly creationTimestamp: Date = new Date();
  readonly eventsLogged = 0;

  addTrace(): void {}
  getTraceMessage(): ChannelTrace {
    return {
      creation_timestamp: dateToProtoTimestamp(this.creationTimestamp),
      num_events_logged: this.eventsLogged,
      events: [],
    };
  }
}

export class ChannelzTrace {
  events: TraceEvent[] = [];
  creationTimestamp: Date;
  eventsLogged = 0;

  constructor() {
    this.creationTimestamp = new Date();
  }

  addTrace(
    severity: TraceSeverity,
    description: string,
    child?: ChannelRef | SubchannelRef
  ) {
    const timestamp = new Date();
    this.events.push({
      description: description,
      severity: severity,
      timestamp: timestamp,
      childChannel: child?.kind === 'channel' ? child : undefined,
      childSubchannel: child?.kind === 'subchannel' ? child : undefined,
    });
    // Whenever the trace array gets too large, discard the first half
    if (this.events.length >= TARGET_RETAINED_TRACES * 2) {
      this.events = this.events.slice(TARGET_RETAINED_TRACES);
    }
    this.eventsLogged += 1;
  }

  getTraceMessage(): ChannelTrace {
    return {
      creation_timestamp: dateToProtoTimestamp(this.creationTimestamp),
      num_events_logged: this.eventsLogged,
      events: this.events.map(event => {
        return {
          description: event.description,
          severity: event.severity,
          timestamp: dateToProtoTimestamp(event.timestamp),
          channel_ref: event.childChannel
            ? channelRefToMessage(event.childChannel)
            : null,
          subchannel_ref: event.childSubchannel
            ? subchannelRefToMessage(event.childSubchannel)
            : null,
        };
      }),
    };
  }
}

type RefOrderedMap = OrderedMap<
  number,
  { ref: { id: number; kind: EntityTypes; name: string }; count: number }
>;

export class ChannelzChildrenTracker {
  private channelChildren: RefOrderedMap = new OrderedMap();
  private subchannelChildren: RefOrderedMap = new OrderedMap();
  private socketChildren: RefOrderedMap = new OrderedMap();
  private trackerMap = {
    [EntityTypes.channel]: this.channelChildren,
    [EntityTypes.subchannel]: this.subchannelChildren,
    [EntityTypes.socket]: this.socketChildren,
  } as const;

  refChild(child: ChannelRef | SubchannelRef | SocketRef) {
    const tracker = this.trackerMap[child.kind];
    const trackedChild = tracker.find(child.id);

    if (trackedChild.equals(tracker.end())) {
      tracker.setElement(
        child.id,
        {
          ref: child,
          count: 1,
        },
        trackedChild
      );
    } else {
      trackedChild.pointer[1].count += 1;
    }
  }

  unrefChild(child: ChannelRef | SubchannelRef | SocketRef) {
    const tracker = this.trackerMap[child.kind];
    const trackedChild = tracker.getElementByKey(child.id);
    if (trackedChild !== undefined) {
      trackedChild.count -= 1;
      if (trackedChild.count === 0) {
        tracker.eraseElementByKey(child.id);
      }
    }
  }

  getChildLists(): ChannelzChildren {
    return {
      channels: this.channelChildren as ChannelzChildren['channels'],
      subchannels: this.subchannelChildren as ChannelzChildren['subchannels'],
      sockets: this.socketChildren as ChannelzChildren['sockets'],
    };
  }
}

export class ChannelzChildrenTrackerStub extends ChannelzChildrenTracker {
  override refChild(): void {}
  override unrefChild(): void {}
}

export class ChannelzCallTracker {
  callsStarted = 0;
  callsSucceeded = 0;
  callsFailed = 0;
  lastCallStartedTimestamp: Date | null = null;

  addCallStarted() {
    this.callsStarted += 1;
    this.lastCallStartedTimestamp = new Date();
  }
  addCallSucceeded() {
    this.callsSucceeded += 1;
  }
  addCallFailed() {
    this.callsFailed += 1;
  }
}

export class ChannelzCallTrackerStub extends ChannelzCallTracker {
  override addCallStarted() {}
  override addCallSucceeded() {}
  override addCallFailed() {}
}

export interface ChannelzChildren {
  channels: OrderedMap<number, { ref: ChannelRef; count: number }>;
  subchannels: OrderedMap<number, { ref: SubchannelRef; count: number }>;
  sockets: OrderedMap<number, { ref: SocketRef; count: number }>;
}

export interface ChannelInfo {
  target: string;
  state: ConnectivityState;
  trace: ChannelzTrace | ChannelzTraceStub;
  callTracker: ChannelzCallTracker | ChannelzCallTrackerStub;
  children: ChannelzChildren;
}

export type SubchannelInfo = ChannelInfo;

export interface ServerInfo {
  trace: ChannelzTrace;
  callTracker: ChannelzCallTracker;
  listenerChildren: ChannelzChildren;
  sessionChildren: ChannelzChildren;
}

export interface TlsInfo {
  cipherSuiteStandardName: string | null;
  cipherSuiteOtherName: string | null;
  localCertificate: Buffer | null;
  remoteCertificate: Buffer | null;
}

export interface SocketInfo {
  localAddress: SubchannelAddress | null;
  remoteAddress: SubchannelAddress | null;
  security: TlsInfo | null;
  remoteName: string | null;
  streamsStarted: number;
  streamsSucceeded: number;
  streamsFailed: number;
  messagesSent: number;
  messagesReceived: number;
  keepAlivesSent: number;
  lastLocalStreamCreatedTimestamp: Date | null;
  lastRemoteStreamCreatedTimestamp: Date | null;
  lastMessageSentTimestamp: Date | null;
  lastMessageReceivedTimestamp: Date | null;
  localFlowControlWindow: number | null;
  remoteFlowControlWindow: number | null;
}

interface ChannelEntry {
  ref: ChannelRef;
  getInfo(): ChannelInfo;
}

interface SubchannelEntry {
  ref: SubchannelRef;
  getInfo(): SubchannelInfo;
}

interface ServerEntry {
  ref: ServerRef;
  getInfo(): S