# Accounts API Deployment Guide for cPanel

## Prerequisites

✅ **Completed:**
- PostgreSQL database created
- Auth0 credentials configured
- JWT signing key generated
- Environment variables file created (`.env`)

⏳ **Still Needed:**
- SMTP configuration (for sending emails)

## Step 1: Install Dependencies

```bash
cd accounts/api
npm install
```

## Step 2: Configure SMTP

Edit `.env` file and add your SMTP credentials:

```env
SMTP_HOST=smtp.example.com
SMTP_PORT=587
SMTP_USERNAME=your-username
SMTP_PASSWORD=your-password
CONTACT_FORM_DESTINATION=your-email@arslan10227.site
```

**Common SMTP Options:**
- **cPanel Email**: Use your cPanel email SMTP settings
- **Gmail**: `smtp.gmail.com:587` (requires app password)
- **Mailtrap** (for testing): `smtp.mailtrap.io:2525`

## Step 3: Build the Application

```bash
npm run build
```

This compiles TypeScript to JavaScript in the `lib/` directory.

## Step 4: Test Database Connection

Before deploying, test the database connection:

```bash
# Set environment variables
export $(cat .env | grep -v '^#' | xargs)

# Test connection (if you have a test script)
node -e "
const { Pool } = require('pg');
const pool = new Pool({ connectionString: process.env.DATABASE_URL });
pool.query('SELECT NOW()', (err, res) => {
  if (err) {
    console.error('Database connection failed:', err);
    process.exit(1);
  } else {
    console.log('Database connected successfully:', res.rows[0]);
    pool.end();
  }
});
"
```

## Step 5: Run Database Migrations

The migrations run automatically when the server starts, but you can also run them manually:

```bash
# The server will run migrations on startup
npm start
```

Or if you have a migration script:

```bash
npm run migrate
```

## Step 6: Deploy to cPanel

### Option A: Using Node.js App in cPanel

1. **Create Node.js App in cPanel:**
   - Go to cPanel → Node.js Selector
   - Click "Create Application"
   - Node.js version: Select latest (v20+)
   - Application mode: Production
   - Application root: `/home/username/accounts-api` (or your path)
   - Application URL: `accounts.arslan10227.site`
   - Application startup file: `src/server.ts` (or `lib/server.js` if built)

2. **Set Environment Variables:**
   - In Node.js App settings, add all variables from `.env`
   - Or create `.env` file in the application root

3. **Install Dependencies:**
   - Run `npm install` in the application directory
   - Or use cPanel's npm install feature

4. **Start Application:**
   - Click "Run npm start" or "Reload App"

### Option B: Using SSH/Command Line

If you have SSH access:

```bash
# Navigate to your application directory
cd ~/accounts-api

# Install dependencies
npm install --production

# Build the application
npm run build

# Set environment variables (or use .env file)
export $(cat .env | grep -v '^#' | xargs)

# Start with PM2 (recommended for production)
npm install -g pm2
pm2 start lib/server.js --name accounts-api
pm2 save
pm2 startup  # Follow instructions to enable auto-start
```

### Option C: Using cPanel Application Manager

1. Upload files via File Manager or FTP
2. Create Node.js application in cPanel
3. Configure environment variables
4. Start the application

## Step 7: Configure Reverse Proxy (if needed)

If cPanel doesn't handle Node.js directly, you may need to set up a reverse proxy:

### Using .htaccess (Apache)

Create `.htaccess` in your public_html or domain directory:

```apache
RewriteEngine On
RewriteRule ^(.*)$ http://localhost:4000/$1 [P,L]
```

### Using Nginx (if available)

```nginx
server {
    listen 80;
    server_name accounts.arslan10227.site;

    location / {
        proxy_pass http://localhost:4000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
```

## Step 8: Set Up SSL Certificate

1. **Via cPanel:**
   - Go to SSL/TLS Status
   - Select `accounts.arslan10227.site`
   - Click "Run AutoSSL" or install Let's Encrypt certificate

2. **Update .env** to use HTTPS URLs if needed

## Step 9: Test the API

```bash
# Health check
curl https://accounts.arslan10227.site/api/health

# Test CORS
curl -H "Origin: https://arslan10227.site" \
     -H "Access-Control-Request-Method: GET" \
     -X OPTIONS \
     https://accounts.arslan10227.site/api/get-prices

# Test pricing endpoint
curl https://accounts.arslan10227.site/api/get-prices
```

## Step 10: Monitor and Maintain

### Check Logs

```bash
# If using PM2
pm2 logs accounts-api

# If using cPanel Node.js
# Check logs in cPanel → Node.js → View Logs

# Application logs
tail -f ~/accounts-api/logs/app.log
```

### Restart Application

```bash
# PM2
pm2 restart accounts-api

# cPanel Node.js
# Use "Reload App" button in cPanel
```

## Troubleshooting

### Database Connection Issues

- Verify PostgreSQL is running: `psql -U maliklaptops_user -d maliklaptops_httptoolkit`
- Check DATABASE_URL format
- Verify user permissions
- Check firewall rules

### Port Already in Use

- Change PORT in `.env` to an available port
- Update reverse proxy configuration
- Check what's using the port: `lsof -i :4000` or `netstat -tulpn | grep 4000`

### CORS Errors

- Verify ORIGIN_REGEX in `.env` matches your domain
- Check that requests include proper Origin header
- Review server logs for CORS warnings

### Migration Errors

- Check database user has CREATE TABLE permissions
- Verify PostgreSQL version (should be 12+)
- Review migration logs

## Security Checklist

- [ ] `.env` file is not publicly accessible (should be outside public_html)
- [ ] Database password is strong
- [ ] JWT signing key is secure
- [ ] SSL certificate is installed
- [ ] CORS is properly configured
- [ ] Rate limiting is enabled (built into the API)
- [ ] Logs don't contain sensitive information

## Next Steps

After accounts API is deployed:

1. **Build and deploy UI** with `ACCOUNTS_API=https://accounts.arslan10227.site/api`
2. **Configure DNS** for both domains
3. **Test complete authentication flow**
4. **Set up monitoring** (optional)
